program Hex2VKP;
{$I-}
{$APPTYPE CONSOLE}
uses
   SysUtils,IniFiles;

type
   PText =  ^TextFile;
   bytefile = file of Byte;
   PBytefile = ^bytefile;

   tffblock =
      record
         Addr,off,len : integer;
      end;

   tffblocks = array of tffblock;

const
   ErrFileName    : string = '';
   VKPFileName    : string = '';
   InputFileName  : string = '';
   CfgFileName    : string = 'HEX2VKP.INI';
   FlashFileName  : string = '';

   InputFile      : PText = nil;
   OutErrFile     : PText = nil;
   OutVkpFile     : PText = nil;
   Address        : integer = 0;
   FFFile         : PBytefile = nil;
var
   aOutErrFile, aOutVkpFile, aInputFile : System.Text;
   aFlashFile  : Bytefile;

   ErrMsg : string;
   ffblocks : tffblocks;

procedure PrintHelp;
begin
   WriteLn('Intel .Hex to VKlay .VKP Converter. V. 0.1a (C) Redkin 2003');
   WriteLn('USAGE: HEX2VKP [<switches>] file.hex  [file.vkp]');
   WriteLn;
   WriteLn('   file.hex   - input .hex file');
   WriteLn('   file.vkp   - output .vkp file ');
   WriteLn;
   WriteLn('Switches:');
   WriteLn('   -Efilename    - write Errors to filename         (stdout by default)');
   WriteLn('   -Cfilename    - Use configuration file filename  (',CfgFileName,' by default)');
   WriteLn('                 * See ',CfgFileName,' for more info');
   WriteLn('   -Amemaddr     - Use Start address memaddr        (get from .hex by default)');
   WriteLn('   -Fflash.bin   - Use flash.bin for creating UNDO PATCH (no UNDO  by default)');
   WriteLn('   -H            - print this help screen');
   WriteLn;
end;

procedure OpenFFFile;
begin
   if ErrMsg = '' then
      if FlashFileName <> '' then
      begin
         AssignFile(aFlashFile, FlashFileName);
         Reset(aFlashFile);
         if IOResult <> 0 then
            ErrMsg := 'Couldn''t open file '+FlashFileName
         else FFFile := @aFlashFile;
      end;
end;
procedure CloseFFFile;
begin
   if (FlashFileName <> '') and (FFFile <> nil) then
   begin
      CloseFile(FFFile^);
      FFFile := nil;
   end;
end;

function GetFFContent(offs,len : integer) : string;
var
   B  : byte;
   i  : integer;
   S  : string;
begin
   Result := '';
   S := '';
   if ErrMsg <> '' then Exit;
   if FFFile <> nil then
   begin
      Seek(FFFile^,offs);
      if IOResult <> 0 then
      begin
         ErrMsg := 'Invalid Fullflash file'+FlashFileName;
         Exit;
      end;
      for i := 1 to Len do
      begin
         Read(FFFile^,B);
         if IOResult <> 0 then
         begin
            ErrMsg := 'Invalid Fullflash file'+FlashFileName;
            Exit;
         end;
         S := S + IntToHex(B,2);
      end;
      Result := S;
   end;
end;

procedure ReadCFG(FileName : string);
var
   Ini   : TiniFile;
   i     : Integer;
   sec   : string;
begin
   try
      try
         Ini := TIniFile.Create(ExpandFileName(FileName));
      except
         ErrMsg := 'Cannot open '+FileName;
         Exit;
      end;
   SetLength(FfBlocks,0);
   i := 1;
   repeat
      sec := 'Block'+IntToStr(i);
      if Ini.SectionExists(sec) then
      begin
         SetLength(FFBlocks,I);
         FFBlocks[I-1].off := Ini.ReadInteger(sec,'FFOffset',0);
         FFBlocks[I-1].len := Ini.ReadInteger(sec,'Length',0);
         FFBlocks[I-1].Addr := Ini.ReadInteger(sec,'Addr',0);
         Inc(i);
      end
      else
         break;
   until False;
   finally
      FreeAndNil(Ini);
   end;
end;

function GetFFOffset(addr : integer) : integer;
var
   i   : integer;
begin
   Result := -1;
   for i := Low(FFBlocks) to High(FFBlocks) do
   begin
      if (addr >= FFBlocks[i].Addr) and
         (addr <  FFBlocks[i].Addr + FFBlocks[i].len) then
      begin
         Result := addr - FFBlocks[i].Addr + FFBlocks[i].off;
         Exit;
      end;
   end;
end;

procedure OpenErrorFile;
begin
   if ErrFileName = '' then
      OutErrFile := @System.output
   else
   begin
      AssignFile(aOutErrFile, ErrFileName);
      Rewrite(aOutErrFile);
      if IOResult <> 0 then
      begin
         ErrMsg := 'Couldn''t open file '+ErrFileName;
         exit
      end
      else OutErrFile := @aOutErrFile;
   end;
end;

procedure CloseErrorFile;
begin
   if (ErrFileName <> '') and (OutErrFile <> nil) then
   begin
      CloseFile(OutErrFile^);
      OutErrFile := nil;
   end;
end;

procedure OpenInputFile;
begin
   if ErrMsg = '' then
      if InputFileName <> '' then
      begin
         AssignFile(aInputFile, InputFileName);
         Reset(aInputFile);
         if IOResult <> 0 then
            ErrMsg := 'Couldn''t open file '+InputFileName
         else InputFile := @aInputFile;
      end;
end;

procedure CloseInputFile;
begin
   if (InputFileName <> '') and (InputFile <> nil) then
   begin
      CloseFile(InputFile^);
      InputFile := nil;
   end;
end;

procedure OpenVKPFile;
begin
   if (ErrMsg = '') then
   begin
      if VKPFileName = '' then
         VKPFileName := ChangeFileExt(InputFileName,'.vkp');

      AssignFile(aOutVKPFile, VKPFileName);
      Rewrite(aOutVKPFile);

      if IOResult <> 0 then
         ErrMsg := 'Couldn''t write to '+VKPFileName
      else OutVKPFile := @aOutVKPFile;
   end;
end;

procedure CloseVKPFile;
begin
   if (VKPFileName <> '') and (OutVKPFile <> nil) then
   begin
      CloseFile(OutVKPFile^);
      OutVKPFile := nil;
   end;
end;

function Hex2Int(S : string) : integer;
var
   i,N   : integer;
begin
   Result := 0;
   N := 0;
   for i := 1 to Length(S) do
   begin
      N := N shl 4;
      if s[i] in ['0'..'9'] then
          N := N + Ord(S[i]) - Ord('0')
      else if s[i] in ['A'..'F'] then
          N := N + Ord(S[i]) - Ord('A') + 10
      else if s[i] in ['a'..'f'] then
          N := N + Ord(S[i]) - Ord('a') + 10
      else
         Exit;
   end;
   Result := N;
end;

procedure ParseCommandLine;
var
   I  : Integer;
   S  : string;
begin
   if ParamCount = 0 then
   begin
      PrintHelp;
      Halt(2);
   end;

   for I := 1 to ParamCount do
   begin
      S := ParamStr(I);
      if (Length(S) >= 2) and (S[1] in ['-','/']) then
         case UpCase(S[2]) of
            'E' :
               ErrFileName := Copy(S, 3, 255);
            'C' :
               CfgFileName := Copy(S, 3, 255);
            'F' :
            begin
               FlashFileName := Copy(S, 3, 255);
            end;
            'A' :
            begin
               Address := Hex2Int(Copy(S, 3, 255));
               if Address < 0 then
                  ErrMsg := 'Invalid MemAddress: '+Copy(S, 3, 255);
            end;
            'H','?' :
               begin
                  PrintHelp;
                  Halt(0);
               end;
            else
            begin
               ErrMsg := 'Invalid switch: '+S;
            end
         end
      else
      begin
         if InputFileName = '' then
            InputFileName := S
         else if VkpFileName = '' then
            VkpFileName := S
         else
            ErrMsg := 'Invalid command line';
      end;
   end;
end;

procedure OpenFiles;
begin
   if ErrMsg <> '' then exit;
   if InputFileName = '' then
      ErrMsg := 'Incorrect use of switches';
   if ErrMsg <> '' then exit;

   ReadCfg(CfgFileName);
   if ErrMsg <> '' then exit;

   OpenInputFile;
   if ErrMsg <> '' then exit;

   OpenFFFile;
   if ErrMsg <> '' then exit;

   OpenVkpFile;
   if ErrMsg <> '' then exit;
end;

procedure WriteVkpLine(Addr,Undo,Patch : string);
begin
   if Undo <> '' then
      WriteLn(OutVkpFile^, Addr,': ',Undo,' ',Patch)
   else
      WriteLn(OutVkpFile^, Addr,': ',Patch);
end;


procedure Convert;
var
   S,S1,undo  : string;
   Len,Addr,Addr1,CRC,Mode,Segm : Integer;
   HexEnd   : Boolean;
   SegmentAddress : integer;

   function GetHex(var S : string; Len : Integer) : integer;
   var
      j   : integer;
      SS    : string;
   begin
      Result := -1;
      SS := Copy(S,1,Len*2);
      S  := Copy(S,Len*2+1,Length(S));
      if Length(SS) <> Len*2 then Exit;
      j := Hex2Int(SS);
      if j <0 then Exit;
      Result := j;
   end;

begin
   if ErrMsg <> ''  then Exit;
   HexEnd := False;
   SegmentAddress := 0;
   if Address > 0 then SegmentAddress := address;

   while not eof(InputFile^) do
   begin
      Readln(InputFile^,S1);
      if S1 = '' then continue;
      S := S1;
      if S[1] <> ':' then
      begin
         ErrMsg := 'Invalid Line: '+S1;
         Exit;
      end;
      CRC := Hex2Int(Copy(S,Length(S)-1,2)); //now crc is ignored.
      S := Copy(S,2,Length(S)-3);
      Len := GetHex(S,1);
      if Len < 0 then
      begin
         ErrMsg := 'Invalid Length in Line: '+S1;
         Exit;
      end;
      Addr := GetHex(S,2);
      if Addr < 0 then
      begin
         ErrMsg := 'Invalid address in Line: '+S1;
         Exit;
      end;
      Mode := GetHex(S,1);
      case Mode of
      $01:
         begin
            HexEnd := True;
            GetHex(S,Len);
         end;
      $00:
         begin
            if Length(S) <> Len*2 then
            begin
               ErrMsg := 'Invalid Length in Line: '+S1;
               Exit;
            end;
            if SegmentAddress <= 0 then
            begin
               ErrMsg := 'Cannot determine segment address. Use -A switch: '+S1;
               Exit;
            end;

            Addr1 := GetFFOffset(SegmentAddress + addr);
            if Addr1 < 0 then
            begin
               ErrMsg := 'Cannot get offset for address '+IntToHex(SegmentAddress + addr,8);
               Exit;
            end;

            Undo := GetFFContent(Addr1,Len);
            WriteVKPLine(IntToHex(Addr1,8),Undo,S);
            S := '';
         end;
      $04:
         begin
            Segm := GetHex(S,Len);
            if Segm < 0 then
            begin
               ErrMsg := 'Invalid segment in Line: '+S1;
               Exit;
            end;
            if Address > 0 then
            begin
               ErrMsg := '.Hex contains Segment address. Cannot use -A switch:'+S1;
               Exit;
            end;
            SegmentAddress := Segm shl 16;
            Writeln(OutVKPFile^,';---- New block. Segment address:',IntToHex(SegmentAddress,2),'--------')
         end;
      else
         begin
            ErrMsg := 'Don''t know what to do. Ignored: '+S1;
            S := '';
         end;
      end;
      if S <> '' then
      begin
         ErrMsg := 'Invalid Length in Line: '+S1;
         Exit;
      end;
      if HexEnd then break;
   end;
   if not HexEnd then
      begin
         ErrMsg := 'No :00000001FF Hex End Line';
         Exit;
      end;
end;

procedure PrintOutError;
begin
   OpenErrorFile;
   if OutErrFile <> nil then
      WriteLn(OutErrFile^, ErrMsg)
   else Halt(1);
end;

begin
   ErrMsg := '';

   CfgFileName    := IncludeTrailingBackslash(ExtractFilePath(ParamStr(0)))+CfgFileName;

   ParseCommandLine;

   OpenFiles;

   if ErrMsg = '' then
      Convert;
   CloseVkpFile;
   CloseInputFile;
   CloseFFFile;

   if ErrMsg = '' then
      ErrMsg := InputFileName+' -> '+VKPFileName+' Converted OK';
   if ErrMsg <> '' then
   begin
      PrintOutError;
      CloseErrorFile;
   end;
end.
